/**
 * Tooltip Public
 * Version 0.0.3
 *
 * The Custom Component creates a Tooltip. A Tooltip provides additional information or clarification about an interface element.
 *
 * API:
 * appearDelay: number - delay before starting the tooltip animation.
 * appearDuration: number - tooltip fade-in animation duration.
 * showDuration: number - the duration of displaying a Tooltip.
 * disappearDuration: number - tooltip fade-out animation duration.
 * show(): void - starts Tooltip.
 * hide(): void - hides Tooltip.
 * forceHide(): void -  forcefully hides the tooltip without the fade-out animation.
 * (read-only) direction: string - returns the tooltip’s alignment.
 * label: string - the tooltip's text;
 * visible(): boolean - returns whether the tooltip is active at the moment.
 *
 * Events:
 * onTooltipStart - triggers when the appearance animation started.
 * onTooltipFinish - triggers when the disappearance animation ended.
 **/

enum Direction {
    Left,
    Right,
    Top,
    Bottom
}

import { Event } from './Modules/Event/Event';
import { BehaviorEventCallbacks, CallbackType, CustomFunctions } from './Modules/BehaviorSupport/BehaviorEventCallbacks';
import { Utils } from './Modules/Utils';

import UnitType = Canvas.UnitType;
@component
export class Tooltip extends BaseScriptComponent {

    @input
    private autostart: boolean = true;

    @input('string')
    private text: string;

    @input('int')
    @widget(new ComboBoxWidget()
        .addItem('Left', 0)
        .addItem('Right', 1)
        .addItem('Top', 2)
        .addItem('Bottom', 3))
    private tooltipDirection: Direction;

    @ui.group_start('Duration')

    @input('float', '0')
    @label('Delay')
        appearDelay: number;

    @input('float', '0.8')
    @label('Fade In')
        appearDuration: number;

    @input('float', '2.0')
    @label('Shown Time')
        showDuration: number;

    @input('float', '0.8')
    @label('Fade Out')
        disappearDuration: number;

    @ui.group_end

    @input()
    private tooltipPrefabWorld: ObjectPrefab;

    @input()
    private tooltipPrefabPixelsPoints: ObjectPrefab;

    @ui.separator

    @input
    private eventCallbacks: boolean;

    @ui.group_start('Event Callbacks')
    @showIf('eventCallbacks')

    @input('int', '0')
    @widget(new ComboBoxWidget()
        .addItem('None', 0)
        .addItem('Behavior Script', 1)
        .addItem('Behavior Custom', 2)
        .addItem('Custom Function', 3))
    private callbackType: CallbackType;

    @input
    @showIf('callbackType', 1)
    private onTooltipStartBehaviors: ScriptComponent[];

    @input
    @showIf('callbackType', 1)
    private onTooltipFinishBehaviors: ScriptComponent[];

    @input
    @showIf('callbackType', 2)
    private onTooltipStartCustomTriggers: string[];

    @input
    @showIf('callbackType', 2)
    private onTooltipFinishCustomTriggers: string[];

    @input
    @showIf('callbackType', 3)
    @allowUndefined
    private onTooltipStartFunctions: CustomFunctions[];

    @input
    @showIf('callbackType', 3)
    @allowUndefined
    private onTooltipFinishFunctions: CustomFunctions[];
    @ui.group_end
        onTooltipStart: Event = new Event();
    onTooltipFinish: Event = new Event();
    private readonly textOffsetWorld: vec2 = new vec2(0.07, 0); //Offset applied to the text component so that it is centered on the background of the tooltip when the Unit Type is set to World.
    private readonly textOffsetPixelsPoints: vec2 = new vec2(5, 0); //Offset applied to the text component so that it is centered on the background of the tooltip when the Unit Type is set to Points or Pixels.
    private readonly intervalCoef: number = 50; //Coefficient applied to the specified start and end intervals of the tooltip relative to the Image component when the Unit Type is set to Points or Pixels.
    private startInterval: vec2 = new vec2(1.0, 1.0);
    private finishInterval: vec2 = new vec2(0.01, 0.01);
    private unitType: UnitType = UnitType.World;
    private startTooltipCenterPosition: vec2 = null;

    private tooltipText: Text = null;
    private targetST: ScreenTransform = null;
    private targetSO: SceneObject = null;
    private targetImage: Image = null;

    private imageExtentsTargetST: ScreenTransform = null;
    private imageExtentsTargetSO: SceneObject = null;

    private tooltipSO: SceneObject = null;
    private tooltipST: ScreenTransform = null;

    private textExtentsTargetST: ScreenTransform = null;
    private textExtentsTargetSO: SceneObject = null;
    private textSO: SceneObject = null;

    private backgroundSO: SceneObject = null;
    private backgroundImage: Image = null;
    private backgroundST: ScreenTransform = null;

    private isShowAnimationPlaying: boolean = false;
    private isHideAnimationPlaying: boolean = false;
    private isShown: boolean = false;
    private isStartFromAPI: boolean = false;
    onAwake() {
        this.initializeEventCallbacks();
        this.initializeTarget();
        this.initializeDestroy();
    }

    /*
     * Returns the direction of the tooltip.
     */

    get direction() {
        switch (this.tooltipDirection) {
            case Direction.Left:
                return 'Left';
            case Direction.Right:
                return 'Right';
            case Direction.Bottom:
                return 'Bottom';
            case Direction.Top:
                return 'Top';
            default:
        }
    }

    set direction(newDirection: 'Left'|'Right'|'Top'|'Bottom') {
        if (this.tooltipDirection === Direction[newDirection]) {
            return;
        }
        this.tooltipDirection = Direction[newDirection];
        switch (this.tooltipDirection) {
            case Direction.Top:
            case Direction.Bottom:
                this.textSO.getComponent('ScreenTransform').offsets.setCenter(vec2.zero());
                break;
            case Direction.Left:
            case Direction.Right:
                const offset = (this.unitType == UnitType.World)
                    ? this.textOffsetWorld.mult(this.directionVector)
                    : this.textOffsetPixelsPoints.mult(this.directionVector);
                this.textSO.getComponent('ScreenTransform').offsets.setCenter(offset);
                break;
        }
        this.setPointerDirection();
    }

    get label() : string {
        return this.tooltipText.text;
    }

    set label(newLabel: string) {
        if (!this.visible) this.tooltipText.text = newLabel;
    }

    get visible() : boolean {
        return (this.isShowAnimationPlaying || this.isShown || this.isHideAnimationPlaying);
    }

    /**
     * Determines the vector of Tooltip placement relative to the selected direction..
     * @private
     *
     */

    private get directionVector() : vec2 {
        switch (this.tooltipDirection) {
            case Direction.Left:
                return vec2.left();
            case Direction.Right:
                return vec2.right();
            case Direction.Bottom:
                return vec2.down();
            case Direction.Top:
                return vec2.up();
            default:
        }

    }
    /*
    * Forcefully hide the tooltip without the fade-out animation.
    */
    forceHide(): void {
        this.setAlphaText(0);
        this.setAlphaImage(0);
        this.onTooltipFinish.trigger();
        this.isShowAnimationPlaying = false;
        this.isHideAnimationPlaying = false;
        this.isShown = false;
        this.isStartFromAPI = false;
    }

    /*
    * Initiates the process of Tooltip disappearance.
    */
    hide(): void {
        if (this.isShowAnimationPlaying || this.isHideAnimationPlaying || !this.isShown) return;
        this.isHideAnimationPlaying = true;
        this.hideAnimation();
    }
    /*
     * Initiates the process of Tooltip appearance.
     */
    show(): void {
        if (this.isShowAnimationPlaying || this.isHideAnimationPlaying || this.isShown) return;
        this.isStartFromAPI = true;
        this.resetSettings();
        this.isShowAnimationPlaying = true;
    }

    /**
     * Adds listeners to `onTooltipStart` and `onTooltipFinish` events.
     * @private
     */
    private initializeEventCallbacks(): void {
        if (this.eventCallbacks && this.callbackType !== CallbackType.None) {
            this.onTooltipStart.add(BehaviorEventCallbacks.invokeCallbackFromInputs(this, 'onTooltipStart'));
            this.onTooltipFinish.add(BehaviorEventCallbacks.invokeCallbackFromInputs(this, 'onTooltipFinish'));
        }
    }

    /**
     * Initializes the target for Tooltip.
     * @private
     */

    private initializeTarget(): void {
        if (isNull(this.getSceneObject().getComponent('ScreenTransform'))) {
            print('Warning: Object doesn\'t have Component.ScreenTransform');
            return;
        }
        if (isNull(this.getSceneObject().getComponent('Image'))) {
            print('Warning: Object doesn\'t have Component.Image');
            return;
        }

        this.targetSO = this.getSceneObject();
        this.targetST = this.targetSO.getComponent('ScreenTransform');
        this.targetImage = this.targetSO.getComponent('Image');

        this.initializeImageExtentsTarget();
        this.initializeUnitType();
        this.instantiateTooltipPrefab();
    }

    private initializeUnitType(): void {
        const cameraSO = Utils.findOrthographicCamera(this.targetSO);
        if (isNull(cameraSO)) {
            return;
        }
        const canvasComponent = cameraSO.getComponent('Component.Canvas');
        if (isNull(canvasComponent)) {
            return;
        }
        switch (canvasComponent.unitType) {
            case UnitType.Pixels:
                this.unitType = UnitType.Pixels;
                this.startInterval = this.startInterval.uniformScale(this.intervalCoef);
                this.finishInterval = this.finishInterval.uniformScale(this.intervalCoef);
                break;
            case UnitType.Points:
                this.unitType = UnitType.Points;
                this.startInterval = this.startInterval.uniformScale(this.intervalCoef);
                this.finishInterval = this.finishInterval.uniformScale(this.intervalCoef);
                break;
            default :
                this.unitType = UnitType.World;
                break;
        }

    }

    /**
     * Instantiating a prefab to create a Tooltip.
     * @private
     */
    private instantiateTooltipPrefab(): void {
        if (this.unitType == UnitType.World) {
            this.tooltipSO = this.tooltipPrefabWorld.instantiate(this.targetSO);
        } else {
            this.tooltipSO = this.tooltipPrefabPixelsPoints.instantiate(this.targetSO);
        }
        this.initializeTooltipObjects();
    }

    /**
     * Initializes tooltip objects.
     * @private
     */
    private initializeTooltipObjects(): void {
        if (this.tooltipSO.getChildrenCount() != 2) {
            print('Warning: Incorrect prefab structure.');
            return;
        }
        this.textExtentsTargetSO = this.tooltipSO.getChild(0);
        this.textSO = this.tooltipSO.getChild(1);
        if (this.textExtentsTargetSO.getChildrenCount() != 1) {
            print('Warning: Incorrect prefab structure.');
            return;
        }
        this.backgroundSO = this.textExtentsTargetSO.getChild(0);

        this.initializeTooltipComponents();
        this.setRenderLayer(this.targetSO.layer);
    }

    /**
     * Initializes tooltip components.
     * @private
     */
    private initializeTooltipComponents(): void {
        this.tooltipST = this.tooltipSO.getComponent('ScreenTransform');

        this.textExtentsTargetST = this.textExtentsTargetSO.getComponent('ScreenTransform');
        this.tooltipText = this.textSO.getComponent('Text');
        this.backgroundImage = this.backgroundSO.getComponent('Image');
        this.cloneAndReplaceMaterial(this.backgroundImage);
        this.backgroundST = this.backgroundSO.getComponent('ScreenTransform');

        if (isNull(this.tooltipST) || isNull(this.textExtentsTargetST) || isNull(this.tooltipText) ||
            isNull(this.backgroundImage) || isNull(this.backgroundST)) {
            print('Warning: Prefab components recognition error.');
            return;
        }

        if (this.unitType == UnitType.World) {
            if (this.tooltipDirection == Direction.Left) this.textSO.getComponent('ScreenTransform').offsets.setCenter(this.textOffsetWorld.mult(this.directionVector));
            if (this.tooltipDirection == Direction.Right) this.textSO.getComponent('ScreenTransform').offsets.setCenter(this.textOffsetWorld.mult(this.directionVector));
        } else {
            if (this.tooltipDirection == Direction.Left) this.textSO.getComponent('ScreenTransform').offsets.setCenter(this.textOffsetPixelsPoints.mult(this.directionVector));
            if (this.tooltipDirection == Direction.Right) this.textSO.getComponent('ScreenTransform').offsets.setCenter(this.textOffsetPixelsPoints.mult(this.directionVector));
        }

        this.setRenderOrder();
        this.setPointerDirection();
        this.setTooltipText();
        this.resetSettings();
    }

    private initializeImageExtentsTarget(): void {
        if (!isNull(this.targetImage.extentsTarget)) {
            this.imageExtentsTargetST = this.targetImage.extentsTarget;
            this.imageExtentsTargetSO = this.imageExtentsTargetST.getSceneObject();
            return;
        }
        this.imageExtentsTargetSO = global.scene.createSceneObject('ImageExtentsTarget');
        this.imageExtentsTargetSO.setParent(this.targetSO);
        this.imageExtentsTargetST = this.imageExtentsTargetSO.createComponent('ScreenTransform');
        this.targetImage.extentsTarget = this.imageExtentsTargetST;
    }

    /**
     * Determines the direction of the pointer.
     * @private
     */

    private setPointerDirection(): void {
        switch (this.tooltipDirection) {
            case Direction.Left:
                this.backgroundImage.getMaterial(0).mainPass.position_js = Direction.Right;
                break;
            case Direction.Right:
                this.backgroundImage.getMaterial(0).mainPass.position_js = Direction.Left;
                break;
            case Direction.Top:
                this.backgroundImage.getMaterial(0).mainPass.position_js = Direction.Top;
                break;
            case Direction.Bottom:
                this.backgroundImage.getMaterial(0).mainPass.position_js = Direction.Bottom;
                break;
            default:
        }

    }

    /**
     * Sets the tooltip text.
     * @private
     */
    private setTooltipText(): void {
        this.tooltipText.text = this.text;
    }

    /**
     * Sets the RenderLayer for tooltip objects relative to their target.
     * @param value
     * @private
     */
    private setRenderLayer(value: LayerSet): void {
        this.tooltipSO.layer = value;
        this.textExtentsTargetSO.layer = value;
        this.textSO.layer = value;
        this.backgroundSO.layer = value;
        this.imageExtentsTargetSO.layer = value;
    }

    /**
     * Sets the RenderOrder for tooltip components relative to their target.
     * @private
     */
    private setRenderOrder(): void {
        const tooltipRenderOrder = this.targetSO.getComponent('Image').getRenderOrder() + 1;
        this.backgroundImage.setRenderOrder(tooltipRenderOrder);
        this.tooltipText.setRenderOrder(tooltipRenderOrder);
    }
    /**
     * Resets settings after Tooltip initialization and before replaying the animation.".
     * @private
     */
    private resetSettings(): void {
        this.setAlphaText(0);
        this.setAlphaImage(0);
        this.isShowAnimationPlaying = false;
        this.isHideAnimationPlaying = false;
        this.isShown = false;

        this.setTooltipPosition();
    }

    /**
     * Sets the initial position of the Tooltip before the start of the appearance animation.
     * @private
     */
    private setTooltipPosition(): void {
        const delay = this.createEvent('DelayedCallbackEvent');
        delay.bind(() => {
            const tooltipSize = this.sizeRelativeTargetST(vec2.one());
            const targetSize = this.imageExtentsTargetST.anchors.getSize();

            let newTooltipCenter = this.imageExtentsTargetST.anchors.getCenter(); //Places the tooltip in the center of the target.
            newTooltipCenter = newTooltipCenter.add(targetSize.uniformScale(0.5).mult(this.directionVector)); //Shifts the tooltip to the desired edge of the target according to the selected direction.
            newTooltipCenter = newTooltipCenter.add(tooltipSize.uniformScale(0.5).mult(this.directionVector));

            this.tooltipST.anchors.setCenter(newTooltipCenter);
            this.tooltipST.offsets.setCenter(this.startInterval.mult(this.directionVector));
            this.startTooltipCenterPosition = newTooltipCenter;
            this.showAnimation();
            this.removeEvent(delay);
        });
        delay.reset(0.1); //A delay is required for the ExtentTarget dimensions to be determined.
    }

    private sizeRelativeTargetST(vector: vec2): vec2 {
        const leftPoint = this.targetST.worldPointToLocalPoint(this.backgroundST.localPointToWorldPoint(new vec2(-vector.x, 0)));
        const rightPoint = this.targetST.worldPointToLocalPoint(this.backgroundST.localPointToWorldPoint(new vec2(vector.x, 0)));
        const topPoint = this.targetST.worldPointToLocalPoint(this.backgroundST.localPointToWorldPoint(new vec2(0, -vector.y)));
        const bottomPoint = this.targetST.worldPointToLocalPoint(this.backgroundST.localPointToWorldPoint(new vec2(0, vector.y)));

        const width = bottomPoint.y - topPoint.y;
        const height = rightPoint.x - leftPoint.x;
        return new vec2(height, width);
    }

    /**
     * Sets the Alpha value for the text.
     * @private
     */

    private setAlphaText(value: number): void {
        const color = this.tooltipText.textFill.color;
        color.a = value;
        this.tooltipText.textFill.color = color;
    }
    /**
     * Sets the Alpha value for the background.
     * @private
     */

    private setAlphaImage(value: number): void {
        const color = this.backgroundImage.getMaterial(0).mainPass.baseColor;
        color.a = value;
        this.backgroundImage.getMaterial(0).mainPass.baseColor = color;
    }

    /**
     * Calculates and sets a new position for the Tooltip during the appearance animation.
     * @private
     */

    private setNewMovePosition(animationValue: number): void {
        const step = this.startInterval.sub(this.finishInterval).uniformScale(animationValue);
        const newOffset = this.startInterval.sub(step).mult(this.directionVector);
        this.tooltipST.offsets.setCenter(newOffset);
    }

    /**
     * Animation of Tooltip disappearance.
     * @private
     */
    private hideAnimation(): void {
        const startTime = getTime();
        const update = this.createEvent('UpdateEvent');
        update.bind(() => {
            if (getTime() - startTime < this.disappearDuration) {
                const animationPlayTime = (getTime() - startTime) / this.disappearDuration;
                this.setAlphaText(1.0 - animationPlayTime);
                this.setAlphaImage(1.0 - animationPlayTime);
            } else {
                this.setAlphaText(0);
                this.setAlphaImage(0);
                this.onTooltipFinish.trigger();
                this.isHideAnimationPlaying = false;
                this.isShown = false;
                this.isStartFromAPI = false;
                update.enabled = false;
                this.removeEvent(update);
            }
        });
    }
    /**
     * Animation of Tooltip appearance.
     * @private
     */

    private showAnimation(): void {
        const delayBeforeAppearAnimation = this.createEvent('DelayedCallbackEvent');
        delayBeforeAppearAnimation .bind(() => {
            if (!this.isStartFromAPI && !this.autostart) return;
            this.autostart = false;
            const startTime = getTime();
            this.onTooltipStart.trigger();
            const update = this.createEvent('UpdateEvent');
            update.bind(() => {
                if (getTime() - startTime < this.appearDuration) {
                    const animationPlayTime = (getTime() - startTime) / this.appearDuration;
                    this.setAlphaText(animationPlayTime);
                    this.setAlphaImage(animationPlayTime);
                    this.setNewMovePosition(animationPlayTime);
                } else {
                    this.setAlphaText(1);
                    this.setAlphaImage(1);
                    this.isShowAnimationPlaying = false;
                    this.isShown = true;
                    update.enabled = false;
                    const delayBeforeHideAnimation = this.createEvent('DelayedCallbackEvent');
                    delayBeforeHideAnimation.bind(() => {
                        this.hide();
                        this.removeEvent(delayBeforeHideAnimation);
                        this.removeEvent(update);
                        this.removeEvent(delayBeforeAppearAnimation);
                    });
                    delayBeforeHideAnimation.reset(this.showDuration);
                }
            });
        });
        delayBeforeAppearAnimation .reset(this.appearDelay);

    }

    private initializeDestroy(): void {
        this.createEvent('OnDestroyEvent').bind(() => {
            if (!isNull(this.tooltipSO) && this.tooltipSO.destroy) this.tooltipSO.destroy();
            if (!isNull(this.backgroundSO) && this.backgroundSO.destroy) this.backgroundSO.destroy();
            if (!isNull(this.textExtentsTargetSO) && this.textExtentsTargetSO.destroy) this.textExtentsTargetSO.destroy();
            if (!isNull(this.textSO) && this.textSO.destroy) this.textSO.destroy();
            if (!isNull(this.imageExtentsTargetSO) && this.imageExtentsTargetSO.destroy) this.imageExtentsTargetSO.destroy();
        });
    }

    private cloneAndReplaceMaterial(meshVisual: MaterialMeshVisual): void {
        const clone = meshVisual.mainMaterial.clone();
        meshVisual.mainMaterial = clone;
    }
}
